<?php
/**
 * Modern Error Handler for PHP 8.2+
 * Compatible with PHP Live! application
 */

if (!defined('MODERN_ERROR_HANDLER')) {
    define('MODERN_ERROR_HANDLER', true);
    
    // Set error reporting for development/production
    if (defined('PHPLIVE_DEBUG') && PHPLIVE_DEBUG) {
        error_reporting(E_ALL);
        ini_set('display_errors', 1);
    } else {
        error_reporting(E_ALL & ~E_DEPRECATED & ~E_STRICT);
        ini_set('display_errors', 0);
    }
    
    // Custom error handler
    function modernErrorHandler($errno, $errstr, $errfile, $errline) {
        if (!(error_reporting() & $errno)) {
            return false;
        }
        
        $errorType = match($errno) {
            E_ERROR => 'Fatal Error',
            E_WARNING => 'Warning',
            E_PARSE => 'Parse Error',
            E_NOTICE => 'Notice',
            E_CORE_ERROR => 'Core Error',
            E_CORE_WARNING => 'Core Warning',
            E_COMPILE_ERROR => 'Compile Error',
            E_COMPILE_WARNING => 'Compile Warning',
            E_USER_ERROR => 'User Error',
            E_USER_WARNING => 'User Warning',
            E_USER_NOTICE => 'User Notice',
            E_STRICT => 'Strict',
            E_RECOVERABLE_ERROR => 'Recoverable Error',
            E_DEPRECATED => 'Deprecated',
            E_USER_DEPRECATED => 'User Deprecated',
            default => 'Unknown Error'
        };
        
        $errorMessage = "[$errorType] $errstr in $errfile on line $errline";
        
        // Log error
        error_log($errorMessage);
        
        // For fatal errors, we might want to show a user-friendly message
        if (in_array($errno, [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
            if (defined('PHPLIVE_DEBUG') && PHPLIVE_DEBUG) {
                echo "<div style='background: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; padding: 15px; margin: 10px; border-radius: 4px;'>";
                echo "<strong>Fatal Error:</strong> $errstr<br>";
                echo "<strong>File:</strong> $errfile<br>";
                echo "<strong>Line:</strong> $errline";
                echo "</div>";
            } else {
                echo "<div style='background: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; padding: 15px; margin: 10px; border-radius: 4px;'>";
                echo "<strong>System Error:</strong> An unexpected error occurred. Please try again later.";
                echo "</div>";
            }
        }
        
        return true;
    }
    
    // Set custom error handler
    set_error_handler('modernErrorHandler');
    
    // Handle fatal errors
    register_shutdown_function(function() {
        $error = error_get_last();
        if ($error !== null && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
            modernErrorHandler($error['type'], $error['message'], $error['file'], $error['line']);
        }
    });
}